import java.io.*;
import java.util.*;
import java.math.*;

class NMR
{

  //Convert CalcData objects to NMRData objects
  //Input is a CalcData object (with labels and shielding constants) and the shielding constant of TMS to use.  Output is an NMRData object (i.e. labels and shifts)
  public static NMRData convertCalcToNMR(CalcData myData, double TMS)
    {
    int i=0;
    int nShifts = myData.nShifts;
    String[] mylabels = myData.labels;
    double[] myshifts = new double[nShifts];
    String myfilename = myData.filename;

    for(i=0;i<nShifts;i++)
      {
      myshifts[i]=((TMS-myData.sigmas[i])/(1-TMS/1000000));
      }

    NMRData myNMRData = new NMRData(mylabels, myshifts, myfilename);
    return myNMRData;
    }



  //Convert ExpData objects to NMRData objects
  //(completes the assignment of experimental resonances by comparison to the specified NMRData object)
  public static NMRData convertExpToNMR(ExpData expData, NMRData calcData)
    {
    int i=0; int j=0; int k=0;
    int nexpShifts = expData.nShifts;
    int ncalcShifts = calcData.nShifts;
    int nLabels = expData.nLabels;
    String myfilename=expData.filename;

    expData.sort();
    calcData.sort();

    String[] calclabels = new String[ncalcShifts];
    for(i=0;i<ncalcShifts;i++)
      {
      calclabels[i]=calcData.labels[i];
      }

    int[] assignTemp = new int[nexpShifts];

    for(i=0;i<nexpShifts;i++)
      {
      assignTemp[i]=expData.assign[i];
      if(assignTemp[i]==0) {assignTemp[i]=1;}
      }

    double[] shifts = new double[nexpShifts];
    String[] labels = new String[nexpShifts];

    for(i=0;i<nexpShifts;i++)  //loop over the experimental shifts
      {
      shifts[i]=expData.shifts[i];

      search:
      for(j=0;j<ncalcShifts;j++)  //loop over the calculated shifts
        {
        for(k=0;k<nLabels;k++)  //loop over the labels of the exp shifts
          {
          if((expData.labels[i][k].compareTo(calclabels[j])==0)||((expData.labels[i][k].compareTo("any")==0)&&(calclabels[j].compareTo("used")!=0)))
            {
            if(assignTemp[i]==1)
              {
              labels[i]=calclabels[j];  
              calclabels[j]="used";
              break search;
              }
            else
              {
              assignTemp[i]--;
              }
            }
          else if((expData.labels[i][k].compareTo(calcData.labels[j])==0)&&(assignTemp[i]!=1))  //(it could have been a match but isn't because calclabels[i] has been changed to "used"
            {
            assignTemp[i]--;
            }
          }
        }
//      System.out.println("i="+i+", "+calclabels[calclabels.length-2]);
      }
    NMRData myNMRData = new NMRData(labels, shifts,myfilename);
    return myNMRData;
    }




  //Convert four NMRData objects (two for calc data and two for exp data) to a PairwiseData object
  //(Involves lining up the shifts and removing those not common to all data sets.  Strategy is to read in all the possible labels, removing duplicates, then fill arrays with shifts.  Then check if all four bits of data have been filled in for a given label)       
  public static PairwiseData convertToPairwise(NMRData nmrDataa, NMRData nmrDatab, NMRData nmrDataA, NMRData nmrDataB)
    {
    int i=0; int j=0; int k=0;
    int nShifts=0;

    String[] filenames = new String[4];
    filenames[0]=nmrDataa.filename;
    filenames[1]=nmrDatab.filename;
    filenames[2]=nmrDataA.filename;
    filenames[3]=nmrDataB.filename;


    String[] labelsTemp = new String[(nmrDataa.nShifts*nmrDatab.nShifts*nmrDataA.nShifts*nmrDataB.nShifts)+1];
    
    for(i=0;i<nmrDataa.nShifts;i++)
      {
      labelsTemp[i]=nmrDataa.labels[i];
      }


    int stop=0;
    for(i=0;i<nmrDatab.nShifts;i++)
      {
      stop=0;
      for(j=0;stop==0;j++)
        {
        if(nmrDatab.labels[i].compareTo(labelsTemp[j])==0) {stop=1;}
        if(labelsTemp[j]==null) { labelsTemp[j]=nmrDatab.labels[i]; stop=1;}
        }
      }
    for(i=0;i<nmrDataA.nShifts;i++)
      {
      stop=0;
      for(j=0;stop==0;j++)
        {
        if(nmrDataA.labels[i].compareTo(labelsTemp[j])==0) {stop=1;}
        if(labelsTemp[j]==null) { labelsTemp[j]=nmrDataA.labels[i]; stop=1;}
        }
      }
    for(i=0;i<nmrDataB.nShifts;i++)
      {
      stop=0;
      for(j=0;stop==0;j++)
        {
        if(nmrDataB.labels[i].compareTo(labelsTemp[j])==0) {stop=1;}
        if(labelsTemp[j]==null) { labelsTemp[j]=nmrDataB.labels[i]; stop=1;}
        }
      }

    for(i=0;labelsTemp[i]!=null;i++) {nShifts++;}

    String[] labels = new String[nShifts];
    double[] shiftsa = new double[nShifts];
    double[] shiftsb = new double[nShifts];
    double[] shiftsA = new double[nShifts];
    double[] shiftsB = new double[nShifts];
    int[] valid = new int[nShifts];

    for(i=0;i<nShifts;i++) {valid[i]=1;}

    for(i=0;i<nShifts;i++)
      {
      labels[i]=labelsTemp[i];
      }

    for(i=0;i<nShifts;i++)
      {
      for(j=0;j<nmrDataa.nShifts;j++)
        {
        if(labels[i].compareTo(nmrDataa.labels[j])==0)
          {
          shiftsa[i]=nmrDataa.shifts[j]; valid[i]=valid[i]*2;
          break;
          }
        }
      }
    for(i=0;i<nShifts;i++)
      {
      for(j=0;j<nmrDatab.nShifts;j++)
        {
        if(labels[i].compareTo(nmrDatab.labels[j])==0)
          {
          shiftsb[i]=nmrDatab.shifts[j]; valid[i]=valid[i]*2;
          break;
          }
        }
      }
    for(i=0;i<nShifts;i++)
      {
      for(j=0;j<nmrDataA.nShifts;j++)
        {
        if(labels[i].compareTo(nmrDataA.labels[j])==0)
          {
          shiftsA[i]=nmrDataA.shifts[j]; valid[i]=valid[i]*2;
          break;
          }
        }
      }
    for(i=0;i<nShifts;i++)
      {
      for(j=0;j<nmrDataB.nShifts;j++)
        {
        if(labels[i].compareTo(nmrDataB.labels[j])==0)
          {
          shiftsB[i]=nmrDataB.shifts[j]; valid[i]=valid[i]*2;
          break;
          }
        }
      }
/*
    for(i=0;i<nShifts;i++)
      {
      System.out.println(labels[i]+" "+shiftsa[i]+" "+shiftsb[i]+" "+shiftsA[i]+" "+shiftsB[i]+" "+valid[i]);
      } */

    int nShifts2=0;
    for(i=0;i<nShifts;i++)
      {
      if(valid[i]==16) {nShifts2++;}
      }

    String[] labels2 = new String[nShifts2];
    double[] shiftsa2 = new double[nShifts2];
    double[] shiftsb2 = new double[nShifts2];
    double[] shiftsA2 = new double[nShifts2];
    double[] shiftsB2 = new double[nShifts2];

    j=0;
    for(i=0;i<nShifts;i++)
      {
      if(valid[i]==16)
        {
        labels2[j]=labels[i];
        shiftsa2[j]=shiftsa[i];
        shiftsb2[j]=shiftsb[i];
        shiftsA2[j]=shiftsA[i];
        shiftsB2[j]=shiftsB[i];
        j++;
        }
      }

/*
    for(i=0;i<nShifts2;i++)
      {
      System.out.println(labels2[i]+" "+shiftsa2[i]+" "+shiftsb2[i]+" "+shiftsA2[i]+" "+shiftsB2[i]);
      }
*/    
    PairwiseData data = new PairwiseData(labels2, shiftsa2, shiftsb2, shiftsA2, shiftsB2, filenames);

    return data;
    }







  //Convert calc files direct to NMRdata
  public static NMRData convertCalcFileToNMRData(String filename, double TMS)
    {
    CalcData calc = new CalcData(filename);
    NMRData nmr = NMR.convertCalcToNMR(calc,TMS);
    return nmr;
    }

/*  //Convert calc files direct to NMRdata
  public static NMRData convertCalcFileToNMRData(String filename, String tmsFile)
    {
    CalcData calc = new CalcData(filename);
    double tms = NMR.getTmsC(tmsFile);
    NMRData nmr = NMR.convertCalcToNMR(calc,tms);
    return nmr;
    }
*/




  //Convert two NMRData (calc) and two ExpData directly to a PairwiseData object
  public static PairwiseData convertToPairwise2(NMRData calca, NMRData calcb, ExpData expA, ExpData expB)
    {
    NMRData expA2 = NMR.convertExpToNMR(expA,calca);
    NMRData expB2 = NMR.convertExpToNMR(expB,calcb);
    PairwiseData pairwise = convertToPairwise(calca, calcb, expA2, expB2);
    return(pairwise);
    }


  //Convert a two calc files, two exp files and a TMS file to a MultiData object
  public static MultiData convertFilesToMultiData(String[] files, char atom)
    {
   //TMS value to use
   double CsigmaTMS=0;
   double HsigmaTMS=0;
   String TMSfile = files[4];
   BufferedReader buffRead;
   String currLine="";
   try
     {
     buffRead = new BufferedReader(new FileReader(new File(TMSfile)));
     currLine = buffRead.readLine();
     CsigmaTMS = Double.valueOf(currLine);
     currLine=buffRead.readLine();
     HsigmaTMS = Double.valueOf(currLine);
     }        
   catch(Exception e) {System.err.println("Error reading TMSfile");}     

   Double sigmaTMS = 0.0;
   if(atom=='C') sigmaTMS=CsigmaTMS;
   else if(atom=='H') sigmaTMS=HsigmaTMS;

    NMRData calcData[] = new NMRData[2];
    for(int j=0;j<2;j++)
      {
      calcData[j]=NMR.convertCalcFileToNMRData(files[j]+"."+atom,sigmaTMS);
      calcData[j].sort();
      }

   ExpData[] expData = new ExpData[2];
   for(int j=0;j<2;j++)
     {
     expData[j]=new ExpData(files[j+2]+"."+atom);
     expData[j].sort();
     }

   CalcExpData[] calcExp = new CalcExpData[2];

   for(int j=0;j<2;j++)
     {
     NMRData temp = NMR.convertExpToNMR(expData[j],calcData[j]);
     calcExp[j] = new CalcExpData(calcData[j],temp);
     }

   MultiData multi = new MultiData(calcExp);

   return multi;
   }    


  //Convert a two calc files, two exp files and a TMS file to a MultiData object option 2
  public static MultiData convertFilesToMultiData(String calc1, String calc2, String exp1, String exp2, String tmsFile, char atom)
    {
    String[] files = new String[5];
    files[0]=calc1;
    files[1]=calc2;
    files[2]=exp1;
    files[3]=exp2;
    files[4]=tmsFile;
    MultiData multi = NMR.convertFilesToMultiData(files,atom);
    return multi;
    }


  //Get TMS
  public static double getTms(String s, char nucleus)
    {
    double toReturn;
    if(nucleus=='C') toReturn = getTmsC(s);
    else if(nucleus=='H') toReturn = getTmsH(s);
    else throw new RuntimeException("Unknown nucleus: "+nucleus);
    return(toReturn);
    }


  //Get TMS C value
  public static double getTmsC(String s)
    {
    String TMSfile = s;
    double CsigmaTMS=0;
    String currLine="";
    BufferedReader buffRead;
   try
     {
     buffRead = new BufferedReader(new FileReader(new File(TMSfile)));
     currLine = buffRead.readLine();
     CsigmaTMS = Double.valueOf(currLine);
     }        
   catch(Exception e) {System.err.println("Error reading TMSfile");}     
    return(CsigmaTMS);
    }

  //Get TMS H value
  public static double getTmsH(String s)
    {
    String TMSfile = s;
    double HsigmaTMS=0;
    String currLine="";
    BufferedReader buffRead;
   try
     {
     buffRead = new BufferedReader(new FileReader(new File(TMSfile)));
     currLine = buffRead.readLine();
     currLine = buffRead.readLine();
     HsigmaTMS = Double.valueOf(currLine);
     }        
   catch(Exception e) {System.err.println("Error reading TMSfile");}     
    return(HsigmaTMS);
    }



// Read P file
    public static double[] Expect(String param, String pfile)
      {
      double[] x = new double[6];
      BufferedReader buffRead;
      String currLine="";
      currLine="";
      int i=0;
      try
        {
        buffRead = new BufferedReader(new FileReader(new File(pfile)));
        while ((currLine.compareTo("end")!=0)&&(currLine.compareTo(param)!=0))
         {
         currLine = buffRead.readLine();
         } 
        currLine = buffRead.readLine();
        for(i=1;i<6;i++)
          {
          currLine = buffRead.readLine();
          x[i]=Double.valueOf(currLine);
          }
        }        
      catch(Exception e) {;}     
      return x;
      }

    public static double[] Standard(String param, String pfile)
      {
      double[] x = new double[6];
      BufferedReader buffRead;
      String currLine="";
      currLine="";
      int i=0;
      try
        {
        buffRead = new BufferedReader(new FileReader(new File(pfile)));
        while ((currLine.compareTo("end")!=0)&&(currLine.compareTo(param)!=0))
         {
         currLine = buffRead.readLine();
         } 
        for(i=0;i<7;i++)
          {
          currLine = buffRead.readLine();
          }
        for(i=1;i<6;i++)
          {
          currLine = buffRead.readLine();
          x[i]=Double.valueOf(currLine);
          }
        }        
      catch(Exception e) {;}     
      return x;
      }


// Read PI file
    public static double[] ExpectPI(String param, String pfile)
      {
      double[] x = new double[3];
      BufferedReader buffRead;
      String currLine="";
      currLine="";
      int i=0;
      try
        {
        buffRead = new BufferedReader(new FileReader(new File(pfile)));
        while ((currLine.compareTo("end")!=0)&&(currLine.compareTo(param)!=0))
         {
         currLine = buffRead.readLine();
         } 
        for(i=1;i<3;i++)
          {
          currLine = buffRead.readLine();
          StringTokenizer st = new StringTokenizer(currLine," ,\t",false);
          x[i]=Double.valueOf(st.nextToken());
          }
        }        
      catch(Exception e) {;}     
      return x;
      }

    public static double[] StandardPI(String param, String pfile)
      {
      double[] x = new double[3];
      BufferedReader buffRead;
      String currLine="";
      currLine="";
      int i=0;
      try
        {
        buffRead = new BufferedReader(new FileReader(new File(pfile)));
        while ((currLine.compareTo("end")!=0)&&(currLine.compareTo(param)!=0))
         {
         currLine = buffRead.readLine();
         } 
        for(i=1;i<3;i++)
          {
          currLine = buffRead.readLine();
          StringTokenizer st = new StringTokenizer(currLine," ,\t",false);
          x[i]=Double.valueOf(st.nextToken());
          x[i]=Double.valueOf(st.nextToken());
          }
        }        
      catch(Exception e) {;}     
      return x;
      }


    public static double PIExpectRight(String param, String PIfile)
      {
      double[] expect = ExpectPI(param,PIfile); 
      return(expect[1]);
      }
    public static double PIExpectWrong(String param, String PIfile)
      {
      double[] expect = ExpectPI(param,PIfile); 
      return(expect[2]);
      }
    public static double PIStdevRight(String param, String PIfile)
      {
      double[] stdev = StandardPI(param,PIfile); 
      return(stdev[1]);
      }
    public static double PIStdevWrong(String param, String PIfile)
      {
      double[] stdev = StandardPI(param,PIfile); 
      return(stdev[2]);
      }


    public static double getPiParam(String param, String PIfile)
      {
      double value = 0.0;
      BufferedReader buffRead;
      String currLine="";
      try
        {
        buffRead = new BufferedReader(new FileReader(new File(PIfile)));
        while (currLine.compareTo(param)!=0)
          {
          currLine = buffRead.readLine();
          } 
        currLine = buffRead.readLine();
        value = Double.valueOf(currLine);
        }
      catch(Exception e) {System.err.println("Error reading PIfile "+PIfile+" to get parameter "+param); e.printStackTrace();}
      return(value);
      }

    public static double[][] getXdp5Classes(String param, String PIfile)
      {
      double[][] toReturn = new double[2][8];
      BufferedReader buffRead;
      String currLine="";
      try
        {
        buffRead = new BufferedReader(new FileReader(new File(PIfile)));
        while (currLine.compareTo(param)!=0)
          {
          currLine = buffRead.readLine();
          } 
        for(int i=0;i<7;i++)
          {
          currLine=buffRead.readLine();
          StringTokenizer st = new StringTokenizer(currLine);
          toReturn[0][i]=Double.valueOf(st.nextToken());
          toReturn[1][i]=Double.valueOf(st.nextToken());
          }
        currLine=buffRead.readLine();
        StringTokenizer st = new StringTokenizer(currLine);
        toReturn[1][7]=Double.valueOf(st.nextToken());
        buffRead.close();
        }
      catch(Exception e) {System.err.println("Error reading PIfile "+PIfile+" to get parameter "+param+"."); e.printStackTrace();}
      return(toReturn);
      }





  //Average an array of ParameterArray2's
  public static ParameterArray2 averageParameterArray2s(ParameterArray2[] valuesArray)
    {
    int nArrays = valuesArray.length;
    int nParams = valuesArray[0].values.length;
    ParameterArray2 toReturn = new ParameterArray2();
    for(int i=0; i<nParams;i++)
      {
      double temp=0.0;
      for(int j=0;j<nArrays;j++)
        {
        temp=temp+valuesArray[j].values[i];
        }
      toReturn.values[i]=temp/nArrays;
      }
    return(toReturn);
    }

  //Stdev an array of ParameterArray2's
  public static ParameterArray2 stdevParameterArray2s(ParameterArray2[] valuesArray)
    {
    int nArrays = valuesArray.length;
    int nParams = valuesArray[0].values.length;
    ParameterArray2 toReturn = new ParameterArray2();
    for(int i=0; i<nParams;i++)
      {
      double[] values = new double[nArrays];
      for(int j=0;j<nArrays;j++)
        {
        values[j]=valuesArray[j].values[i];
        }
      toReturn.values[i]=NMR.stdevNaN(values);
      }
    return(toReturn);
    }














  //Generate asignment matrix
  public static int[][] generateAsn(int ncalcstructs, int nexpstructs)
      {

      int i; int j; int k; int l; int m;
      int comb = nPr(ncalcstructs,nexpstructs);    
      //System.out.println("No of combinations = "+comb);
    
    int[][] asn2 = new int[nexpstructs][comb];
    int[] blocksize = new int[nexpstructs];

    int used=0;

    
    for(i=0;i<nexpstructs;i++)
      {
      l=0;
      // Size of block:
      blocksize[i]=nPr((ncalcstructs-i-1),(nexpstructs-1-i));
    //System.out.println("blocksize="+blocksize[i]);
      //blocksize[i]=factorial(ncalcstructs-i-1)/factorial(ncalcstructs-nexpstructs);
    //System.out.println("blocksize = "+blocksize[i]);
      while(l<comb)
        {
        for(j=0;j<ncalcstructs&&l<comb;j++)
          {
          used=0;
          for(m=0;m<i;m++)
            {
            if(asn2[m][l]==j)
              {
              used=1;
              //System.out.println("Duplicate");
              break;
              }
            }
          if(used==1)
            {
            continue;
            }
          else
            {
            for(k=0;k<blocksize[i];k++)
              {
              asn2[i][l]=j;
              //System.out.println("asn2["+i+"]["+l+"] = "+asn2[i][l]);
              l++;
              }
            }
          }
        }
      }


// Sort the combinations into a better order

    int[][] asn = new int[nexpstructs][comb];
    int currMax = 0;
    int bigger = 0;
    for(l=(comb-1);l>-1;l--)
      {
      for(k=0;k<comb;k++)
        {
        bigger=0;
        for(i=0;i<nexpstructs;i++)
          {
          if(asn2[i][k]>asn2[i][currMax])
            {
            bigger=1;
            break;
            }
          if(asn2[i][k]<asn2[i][currMax])
            {
            bigger=0;
            break;
            }
          }
        if(bigger==1) 
          {
          currMax=k;
          }
        }

      //System.out.println("currMax="+currMax);
      for(i=0;i<nexpstructs;i++)
        {
        asn[i][l]=asn2[i][currMax];
        asn2[i][currMax]=0;
        }
      }

    return asn;
    }




  //Sort an array of doubles into descending order
  public static double[] sortArray(double[] x)
    {
    int i; int j; int biggest; double biggestValue; int smallest=0; double smallestValue=x[0];
    int N = x.length;
    double[] y = new double[N];
    int[] used = new int[N];

    for(i=0;i<N;i++) //find smallest
      {
      if(x[i]<smallestValue)
        {
        smallestValue=x[i];
        smallest=i;
        }
      }
    for(i=0;i<N;i++)
      {
      biggestValue=smallestValue;
      biggest=smallest;  
      for(j=0;j<N;j++)
        {
        if((x[j]>biggestValue)&&(used[j]==0))
          {
          biggestValue=x[j];  
          biggest=j;  
          }
        }
      y[i]=x[biggest];  
      used[biggest]=1;
      }
    
    return(y);
    }
    

  public static double[] sortDescending(double[] x)
    {
    return(sortArray(x));
    }

  public static double[] sortAscending(double[] x)
    {
    int N = x.length;
    for(int j=0;j<N;j++)
      {
      for(int i=0;i<N-1;i++)
        {
        if(x[i]>x[i+1])
          {
          double temp = x[i];
          x[i]=x[i+1];
          x[i+1]=temp;
          }
        }
      }
    return(x);
    }

    

  public static int[] sortAscending(int[] x)
    {
    int N = x.length;
    for(int j=0;j<N;j++)
      {
      for(int i=0;i<N-1;i++)
        {
        if(x[i]>x[i+1])
          {
          int temp = x[i];
          x[i]=x[i+1];
          x[i+1]=temp;
          }
        }
      }
    return(x);
    }




  //Sort an array of strings into alphabetical order
  public static String[] sortStrings(String[] s)
    {
    int N = s.length;
    for(int j=0;j<N;j++)
      {
      for(int i=0;i<N-1;i++)
        {
        if(s[i].compareTo(s[i+1])>0)
          {
          String temp = s[i];
          s[i]=s[i+1];
          s[i+1]=temp;
          }
        }
      }
    return(s);
    }


  //Sort an array of strings into alphabetical order and remove duplicates
  public static String[] sortStringsFilter(String[] s)
    {
    int N = s.length;
    int nDuplicates=0;
    s = NMR.sortStrings(s);
    for(int i=0;i<N-1;i++)
      {
      if(s[i].compareTo(s[i+1])==0) nDuplicates++;
      }
    String[] toReturn = new String[N-nDuplicates];
    int j=0;
    for(int i=0;i<N-1;i++)
      {
      if(s[i].compareTo(s[i+1])!=0)
        {
        toReturn[j]=s[i];
        j++;
        }
      }
    toReturn[N-nDuplicates-1]=s[N-1];
    return(toReturn);
    }
    



  public static void checkForDuplicates(int[] x)
    {
    x = sortAscending(x);
    for(int i=0;i<x.length-1;i++)
      {
      if(x[i]==x[i+1]) throw new RuntimeException("Warning: duplicate conformers requested: "+x[i]);
      }
    }







  //Determine type given an assignment combination of N exp structures and the assignments being made
  public static int getType(int[] ac, int dataA, int dataB, int dataa, int datab)
    {
    int type=3;
    if(ac[dataA]==dataa){type--;}
    if(ac[dataA]==datab){type++;}
    if(ac[dataB]==dataa){type++;}
    if(ac[dataB]==datab){type--;}
    return(type);
    }
          


  //Trim the ".C" off a filename
  public static String fileTrim(String s)
    {
    StringTokenizer st = new StringTokenizer(s," .",false);
    return(st.nextToken());
    }


  //Get the number of lines in a file (before "end" is reached)
  public static int lines(String filename)
    {
    BufferedReader buffRead;
    String currLine="";
    int nShifts=0;

    try
      {
      buffRead = new BufferedReader(new FileReader(new File(filename)));
      while (currLine!=null&&(currLine.compareTo("end"))!=0)
        {
        currLine = buffRead.readLine();
        nShifts++;
        }
      buffRead.close();
      }
    catch(Exception e) {;}
    nShifts--;
    return nShifts;
    }


  //Get the maximum number of tokens in any line of a file (before "end" is reached) 
  public static int getMaxTokens(String filename)
    {
    int nLines = lines(filename);
    BufferedReader buffRead;
    String currLine="";
    int i=0;
    int maxTokens=0;
    try
      {
      buffRead = new BufferedReader(new FileReader(new File(filename)));
      for(i=0;i<nLines;i++)
        {
        currLine=buffRead.readLine();
        StringTokenizer st = new StringTokenizer(currLine," ,\t",false);
        int lineTokens=0;
        String temp;
        while(st.hasMoreTokens())
          {
          temp=st.nextToken();
          lineTokens++;
          }
        if(lineTokens>maxTokens)
          {
          maxTokens=lineTokens;
          }
        }
      buffRead.close();
      }
    catch(Exception e) {;}
    return maxTokens;
    }


  //Convert the lines in a file to strings
  public static String[] getLines(String filename)
    {
    int nLines = lines(filename);
    String[] toReturn = new String[nLines];
    BufferedReader buffRead;
    try
      {
      buffRead = new BufferedReader(new FileReader(new File(filename)));
      for(int i=0;i<nLines;i++)
        {
        toReturn[i]=buffRead.readLine();
        }
      buffRead.close();
      }
    catch(Exception e) {e.printStackTrace();}
    return(toReturn);
    }


  //Convert lines in a string (with \n characters) to strings
  public static String[] getLines2(String inputString)
    {
    StringTokenizer st = new StringTokenizer(inputString,"\n",false);
    String[] array = new String[st.countTokens()];
    for(int i=0;i<array.length;i++)
      {
      array[i]=st.nextToken();
      }
    return(array);
    }

  //Calculate correl
  public static double calculateCorrel(double[] x, double[] y)
    {
    double Sxx = calculateSxy(x,x);  
    double Syy = calculateSxy(y,y);  
    double Sxy = calculateSxy(x,y);  
    return(Sxy/Math.sqrt(Sxx*Syy));  
    }

  //Calculate slope
  public static double calculateSlope(double[] y, double[] x)
    {
    double Sxx = calculateSxy(x,x);
    double Sxy = calculateSxy(x,y);
    return(Sxy/Sxx);
    }

  //Calculate intercept
  public static double calculateIntercept(double[] y, double[] x)
    {
    double slope = calculateSlope(y,x);
    double xBar = average(x);
    double yBar = average(y);
    return(yBar-slope*xBar);
    }

  //Calculate Sxy
  public static double calculateSxy(double[] x, double[] y)
    {
    int i=0;
    int n=x.length;
    double xBar=average(x);
    double yBar=average(y);
    double Sxy=0;
    for(i=0;i<n;i++)
      {
      Sxy=Sxy+(x[i]-xBar)*(y[i]-yBar);
      }
    return(Sxy);
    }

  //Calculate sum of squares
  public static double calculateSumSquares(double[] x)
    {
    int N = x.length;
    double sum = 0.0;
    for(int i=0;i<N;i++)
      {
      sum = sum + x[i]*x[i];
      }
    return(sum);
    }


  //Calculate average
  public static double average(double[] x)
    {
    int i=0;
    int n=x.length;
    double xBar=0;
    for(i=0;i<n;i++)
      {
      xBar=xBar+x[i];
      }
    return(xBar/n);
    }

  //Calculate average ignoring NaNs
  public static double averageNaN(double[] x)
    {
    double[] y = removeNaN(x);
    return(average(y));
    }

  //Calculate stdev ignoring NaNs
  public static double stdevNaN(double[] x)
    {
    double[] y = removeNaN(x);
    return(stdev(y));
    }

  //Calculate stdevp ignoring NaNs
  public static double stdevpNaN(double[] x)
    {
    double[] y = removeNaN(x);
    return(stdevp(y));
    }



  //Calcuate Stdev
  public static double stdev(double[] x)
    {
    double Sxx=calculateSxy(x,x);
    int N = x.length;
    return(Math.sqrt(Sxx/(N-1)));
    }

  //Calculate Stdevp
  public static double stdevp(double[] x)
    {
    double Sxx=calculateSxy(x,x);
    int N = x.length;
    return(Math.sqrt(Sxx/N));
    }


  //Remove NaN's from an array
  public static double[] removeNaN(double[] x)
    {
    int i=0; int k=0;
    int n = x.length;
    int nNotNaN=n;
    for(i=0;i<n;i++)
      {
      if(Double.isNaN(x[i])) {nNotNaN--;}
      }
    double[] noNaN = new double[nNotNaN];
    for(i=0;i<n;i++)
      {
      if(Double.isNaN(x[i])) 
        {
        }
      else 
        {
        noNaN[k]=x[i];
        k++;
        }
      }
    return(noNaN);
    }


  //Join two arrays
  public static double[] join(double[] x, double[] y)
    {
    int i=0; int j=0;
    double[] joined = new double[x.length+y.length];
    for(i=0;i<x.length;i++)
      {
      joined[j]=x[i];
      j++;
      }
    for(i=0;i<y.length;i++)
      {
      joined[j]=y[i];
      j++;
      }
    return(joined);
    }

  //Join an array of arrays
  public static double[] joinArray(double[][] x)
    {
    int i=0; int j=0; int k=0;
    int numArrays = x.length;
    int numElements[] = new int[numArrays];
    int numElementsTot = 0;
    for(i=0;i<numArrays;i++)
      {
      numElementsTot=numElementsTot+x[i].length;
      numElements[i]=x[i].length;
      }
    double[] joined = new double[numElementsTot];
    for(i=0;i<numArrays;i++)
      {
      for(j=0;j<numElements[i];j++)
        {
        joined[k]=x[i][j];
        k++;
        }
      }
    return(joined);
    }
      
  //Normalise an array of probabilities
  public static double[] normalise(double[] x)
    {
    int i=0;
    int length = x.length;
    double sumProb = 0;
    double[] y = new double[length];
    for(i=0;i<length;i++)
      {
      sumProb=sumProb+x[i];
      }
    for(i=0;i<length;i++)
      {
      y[i]=x[i]/sumProb*100;
      }
    return(y);
    }




  //Calculate MAE
  public static double calculateMae(double[] calc, double[] exp)
    {
    int i;
    double mae=0;
    for(i=0;i<calc.length;i++)
      {
      mae=mae+Math.abs(calc[i]-exp[i]);
      }
    return(mae/calc.length);
    }


  //Calculate MAEprime
  public static double calculateMaePrime(double[] calc, double[] exp)
    {
    int i;
    int nShifts = calc.length;
    double slope=calculateSlope(calc,exp);  
    double intercept=calculateIntercept(calc,exp);   
    double maePrime=0;
    for(i=0;i<nShifts;i++)
      {
      maePrime=maePrime+Math.abs(((calc[i]-intercept)/slope)-exp[i]);
      }
    return(maePrime/nShifts);
    }

   //Calculate wMae
   public static double calculateWMae(double[] calc, double[] exp, double[] weights)
     {
     int i;
     double wMae=0;
     double sumWeights=0;
     for(i=0;i<calc.length;i++)
       {
       wMae=wMae+Math.abs(calc[i]-exp[i])*weights[i];
       sumWeights=sumWeights+weights[i];
       }
     return(wMae/sumWeights);
     }

  //Calculate wMAEprime
  public static double calculateWMaePrime(double[] calc, double[] exp, double[] weights)
    {
    int i;
    int nShifts = calc.length;
    double slope=calculateSlope(calc,exp);  
    double intercept=calculateIntercept(calc,exp);   
    double maePrime=0;
    double sumWeights=0;
    for(i=0;i<nShifts;i++)
      {
      maePrime=maePrime+Math.abs(((calc[i]-intercept)/slope)-exp[i])*weights[i];
      sumWeights=sumWeights+weights[i];
      }
    return(maePrime/sumWeights);
    }

  //Calculate rmse
  public static double calculateRmse(double[] calc, double[] exp)
    {
    int nShifts = calc.length;
    double rmse = 0;
    for(int i=0;i<nShifts;i++)
      {
      rmse = rmse+(calc[i]-exp[i])*(calc[i]-exp[i]);
      }
    return(Math.sqrt(rmse/nShifts));
    }

  //Calculate crmse
  public static double calculateCrmse(double[] calc, double[] exp)
    {
    int nShifts = calc.length;
    double slope = calculateSlope(calc,exp);
    double intercept = calculateIntercept(calc,exp);
    double crmse = 0;
    for(int i=0;i<nShifts;i++)
      {
      crmse = crmse+((calc[i]-intercept)/slope-exp[i])*((calc[i]-intercept)/slope-exp[i]);
      }
    return(Math.sqrt(crmse/nShifts));
    }

  //Calculate max error
  public static double calculateMaxError(double[] calc, double[] exp)
    {
    int nShifts = calc.length;
    double currMax=0.0;
    for(int i=0;i<nShifts;i++)
      {
      if(Math.abs(calc[i]-exp[i])>currMax)
        {
        currMax=Math.abs(calc[i]-exp[i]);
        }
      }
    return(currMax);
    }


  //Calculate cpd
  public static double calculateCpd(double[] calc, double[] exp, double[] calcAv, double[] expAv)
    {
    int i=0;
    double deltaExp=0;
    double deltaCalc=0;
    double num=0;
    double denom=0;
    int nShifts = calc.length;
    for(i=0;i<nShifts;i++)
      {
      deltaExp=exp[i]-expAv[i];
      deltaCalc=calc[i]-calcAv[i];
      denom=denom+deltaExp*deltaExp;
      num=num+deltaExp*deltaCalc;
      }
    return(num/denom);
    }

  //Calculate cpe
  public static double calculateCpe(double[] calc, double[] exp, double[] calcAv, double[] expAv)
    {
    int i=0;
    double deltaExp=0;
    double deltaCalc=0;
    double num=0;
    double denom=0;
    int nShifts = calc.length;
    for(i=0;i<nShifts;i++)
      {
      deltaExp=exp[i]-expAv[i];
      deltaCalc=calc[i]-calcAv[i];
      denom=denom+deltaExp*deltaExp;
      if((Math.abs(deltaCalc/deltaExp))>1)
        {
        num=num+deltaExp*deltaExp*deltaExp/deltaCalc;
        }
      else
        {
        num=num+deltaExp*deltaCalc;
        }
      }
    return(num/denom);
    }



  //Calculate cph
  public static double calculateCph(double[] calc, double[] exp, double[] calcAv, double[] expAv)
    {
    int i=0;
    double deltaExp=0;
    double deltaCalc=0;
    double num=0;
    double denom=0;
    int nShifts = calc.length;
    for(i=0;i<nShifts;i++)
      {
      deltaExp=exp[i]-expAv[i];
      deltaCalc=calc[i]-calcAv[i];
      denom=denom+deltaExp*deltaExp;
      if((deltaCalc/deltaExp)>1)
        {
        num=num+deltaExp*deltaExp*deltaExp/deltaCalc;
        }
      else
        {
        num=num+deltaExp*deltaCalc;
        }
      }
    return(num/denom);
    }


  //Calculate CDP4 from an array of CalcExpData objects
  public static double[] calculateCdp4(CalcExpData[] calcExpData, double expect, double stdev)
    {
    int nIsomers = calcExpData.length;
    double[] cdp4 = new double[nIsomers];
    for(int i=0;i<nIsomers;i++)
      {
      cdp4[i]=calcExpData[i].calculateCdp4(expect,stdev);
      }
    double sumCdp4 = 0.0;
    for(int i=0;i<nIsomers;i++)
      {
      sumCdp4 = sumCdp4+cdp4[i];
      }
    for(int i=0;i<nIsomers;i++)
      {
      cdp4[i]=cdp4[i]/sumCdp4*100;
      }
    return(cdp4);
    }



  //Calculate combined CDP4 from an array of CalcExpData objects
  public static double[] calculateCdp4(CalcExpData[] calcExpDataC, double expectC, double stdevC, CalcExpData[] calcExpDataH, double expectH, double stdevH)
    {
    if(calcExpDataC.length!=calcExpDataH.length)
      {
      throw new RuntimeException("Error: Numbers of isomers for carbon ("+calcExpDataC.length+") and proton ("+calcExpDataH.length+") data are not the same!");
      }
    int nIsomers = calcExpDataC.length;
    double[] cdp4 = new double[nIsomers];
    for(int i=0;i<nIsomers;i++)
      {
      cdp4[i]=calcExpDataC[i].calculateCdp4(expectC,stdevC)*calcExpDataH[i].calculateCdp4(expectH,stdevH);
      }
    double sumCdp4 = 0.0;
    for(int i=0;i<nIsomers;i++)
      {
      sumCdp4 = sumCdp4+cdp4[i];
      }
    for(int i=0;i<nIsomers;i++)
      {
      cdp4[i]=cdp4[i]/sumCdp4*100;
      }
    return(cdp4);
    }



  //Calculate tCDP4 from an array of CalcExpData objects
  public static double[] calculateTcdp4(CalcExpData[] calcExpData, double expect, double stdev, double degfreed)
    {
    int nIsomers = calcExpData.length;
    double[] tcdp4 = new double[nIsomers];
    for(int i=0;i<nIsomers;i++)
      {
      tcdp4[i]=calcExpData[i].calculateTcdp4(expect,stdev,degfreed);
      }
    double sumTcdp4 = 0.0;
    for(int i=0;i<nIsomers;i++)
      {
      sumTcdp4 = sumTcdp4+tcdp4[i];
      }
    for(int i=0;i<nIsomers;i++)
      {
      tcdp4[i]=tcdp4[i]/sumTcdp4*100;
      }
    return(tcdp4);
    }



  //Calculate combined tCDP4 from an array of CalcExpData objects
  public static double[] calculateTcdp4(CalcExpData[] calcExpDataC, double expectC, double stdevC, double degfreedC, CalcExpData[] calcExpDataH, double expectH, double stdevH, double degfreedH)
    {
    if(calcExpDataC.length!=calcExpDataH.length)
      {
      throw new RuntimeException("Error: Numbers of isomers for carbon ("+calcExpDataC.length+") and proton ("+calcExpDataH.length+") data are not the same!");
      }
    int nIsomers = calcExpDataC.length;
    double[] tcdp4 = new double[nIsomers];
    for(int i=0;i<nIsomers;i++)
      {
      tcdp4[i]=calcExpDataC[i].calculateTcdp4(expectC,stdevC,degfreedC)*calcExpDataH[i].calculateTcdp4(expectH,stdevH,degfreedH);
      }
    double sumTcdp4 = 0.0;
    for(int i=0;i<nIsomers;i++)
      {
      sumTcdp4 = sumTcdp4+tcdp4[i];
      }
    for(int i=0;i<nIsomers;i++)
      {
      tcdp4[i]=tcdp4[i]/sumTcdp4*100;
      }
    return(tcdp4);
    }





  public static int whatsGood(String param)
    {
    int toReturn = 0;
    if(param.compareTo("correlC")==0) toReturn=1;
    else if(param.compareTo("MAEC")==0) toReturn=-1;
    else if(param.compareTo("MAE'C")==0) toReturn=-1;
    else if(param.compareTo("wMAEC")==0) toReturn=-1;
    else if(param.compareTo("wMAE'C")==0) toReturn=-1;
    else if(param.compareTo("CP4C")==0) toReturn=1;
    else if(param.compareTo("CP5C")==0) toReturn=1;
    else if(param.compareTo("CP8C")==0) toReturn=1;
    else if(param.compareTo("correlH")==0) toReturn=1;
    else if(param.compareTo("MAEH")==0) toReturn=-1;
    else if(param.compareTo("MAE'H")==0) toReturn=-1;
    else if(param.compareTo("wMAEH")==0) toReturn=-1;
    else if(param.compareTo("wMAE'H")==0) toReturn=-1;
    else if(param.compareTo("CP4H")==0) toReturn=1;
    else if(param.compareTo("CP5H")==0) toReturn=1;
    else if(param.compareTo("CP8H")==0) toReturn=1;
    else if(param.compareTo("correlCH2gav")==0) toReturn=1;
    else if(param.compareTo("MAECHav")==0) toReturn=-1;
    else if(param.compareTo("MAEprimeCHav")==0) toReturn=-1;
    else if(param.compareTo("wMAEM")==0) toReturn=-1;
    else if(param.compareTo("wMAE'M")==0) toReturn=-1;
    else if(param.compareTo("CP4CHav")==0) toReturn=1;
    else if(param.compareTo("CP5CHav")==0) toReturn=1;
    else if(param.compareTo("CP8CHav")==0) toReturn=1;
    else if(param.compareTo("correlCprob")==0) toReturn=1;
    else if(param.compareTo("MAECprob")==0) toReturn=1;
    else if(param.compareTo("MAE'Cprob")==0) toReturn=1;
    else if(param.compareTo("wMAECProb")==0) toReturn=1;
    else if(param.compareTo("wMAE'CProb")==0) toReturn=1;
    else if(param.compareTo("CP4Cprob")==0) toReturn=1;
    else if(param.compareTo("CP5Cprob")==0) toReturn=1;
    else if(param.compareTo("CP8Cprob")==0) toReturn=1;
    else if(param.compareTo("correlHprob")==0) toReturn=1;
    else if(param.compareTo("MAEHprob")==0) toReturn=1;
    else if(param.compareTo("MAE'Hprob")==0) toReturn=1;
    else if(param.compareTo("wMAEHProb")==0) toReturn=1;
    else if(param.compareTo("wMAE'HProb")==0) toReturn=1;
    else if(param.compareTo("CP4Hprob")==0) toReturn=1;
    else if(param.compareTo("CP5Hprob")==0) toReturn=1;
    else if(param.compareTo("CP8Hprob")==0) toReturn=1;
    else if(param.compareTo("correlMprob")==0) toReturn=1;
    else if(param.compareTo("MAECHprob")==0) toReturn=1;
    else if(param.compareTo("MAE'CHprob")==0) toReturn=1;
    else if(param.compareTo("wMAEMProb")==0) toReturn=1;
    else if(param.compareTo("wMAE'Mprob")==0) toReturn=1;
    else if(param.compareTo("CP4CHprob")==0) toReturn=1;
    else if(param.compareTo("CP5CHprob")==0) toReturn=1;
    else if(param.compareTo("CP8CHprob")==0) toReturn=1;
    else if(param.compareTo("correlNprob")==0) toReturn=1;
    else if(param.compareTo("MAENprob")==0) toReturn=1;
    else if(param.compareTo("MAE'Nprob")==0) toReturn=1;
    else if(param.compareTo("wMAENProb")==0) toReturn=1;
    else if(param.compareTo("wMAE'Nprob")==0) toReturn=1;
    else if(param.compareTo("CP4Nprob")==0) toReturn=1;
    else if(param.compareTo("CP5Nprob")==0) toReturn=1;
    else if(param.compareTo("CP8Nprob")==0) toReturn=1;
    else throw new RuntimeException("whatsGood invalid parameter: "+param);
    return(toReturn);
    }


//Combine C and H values
  public static double getValueM(double valueC, double valueH, String param)
    {
    double valueM;
    if(param.compareTo("correl")==0) valueM = 1-Math.sqrt((1-valueC)*(1-valueH));
    else if(param.compareTo("mae")==0) valueM = Math.sqrt(valueC*valueH);
    else if(param.compareTo("cmae")==0) valueM = Math.sqrt(valueC*valueH);
    else if(param.compareTo("wmae")==0) valueM = Math.sqrt(valueC*valueH);
    else if(param.compareTo("wcmae")==0) valueM = Math.sqrt(valueC*valueH);
    else if(param.compareTo("w2mae")==0) valueM = Math.sqrt(valueC*valueH);
    else if(param.compareTo("w2cmae")==0) valueM = Math.sqrt(valueC*valueH);
    else if(param.compareTo("rmse")==0) valueM = Math.sqrt(valueC*valueH);
    else if(param.compareTo("crmse")==0) valueM = Math.sqrt(valueC*valueH);
    else if(param.compareTo("fcmae")==0) valueM = Math.sqrt(valueC*valueH);
    else if(param.compareTo("wfcmae")==0) valueM = Math.sqrt(valueC*valueH);
    else if(param.compareTo("w2fcmae")==0) valueM = Math.sqrt(valueC*valueH);
    else if(param.compareTo("fcrmse")==0) valueM = Math.sqrt(valueC*valueH);
    else if(param.compareTo("dp2")==0) valueM = Math.sqrt(valueC*valueH);
    else if(param.compareTo("cpd")==0) valueM = 0.5*(valueC+valueH);
    else if(param.compareTo("cpe")==0) valueM = 0.5*(valueC+valueH);
    else if(param.compareTo("cph")==0) valueM = 0.5*(valueC+valueH);
    else throw new RuntimeException("Unknown parameter: "+param);
    return(valueM);
    }





//Convert a file name, eg BEA, to a Latex Sr, eg \Sr{BE}{a}
  public static String convertFileToLatexSr(String structureFile, int numInitialLetters)
    {
    int i=0;
    StringBuffer sb = new StringBuffer();
    sb.append("\\Sr{");
    for(i=0;i<numInitialLetters;i++)
      {
      sb.append(structureFile.charAt(i));
      }
    sb.append("}{");
    if(Character.isLetter(structureFile.charAt(i)))
      {
      if(Character.isUpperCase(structureFile.charAt(i)))
        {
        sb.append(Character.toLowerCase(structureFile.charAt(i)));
        }
      else
        {
        sb.append(structureFile.charAt(i));
        }
      }
    else
      {
      throw new RuntimeException("I don't understand this structure file: "+structureFile +" (initialLetters="+numInitialLetters);
      }
    sb.append("}");
    return(sb.toString());
    }






// Evaluate n!
    public static int factorial( int n )
    {
        if( n <= 1 )     // base case
            return 1;
        else
            return n * factorial( n - 1 );
    }



// Evaluate nPr
    public static int nPr( int n, int r )
    {
      int nPr=1;
      int m=n-r;
      while(n>m)
        {
        nPr=nPr*n;
        n--;
        }
      return nPr;
    }

// Evaluate nCr
   public static int nCr(int n, int r)
     {
     int nCr=nPr(n,r)/factorial(r);
     return(nCr);
     }

// Evaluate Kronecker delta
   public static int kDelta(int i, int j)
     {
     if(i==j) return(1);
     else return(0);
     }




}
