import java.io.*;

class FindQRC {

/**
 * <P>
 * Copyright <A HREF="http://www.ch.cam.ac.uk/CUCL/staff/jmg.html">J M Goodman</A>, University of Cambridge, 2003
 * <P>
 *
 * This class reads a Jaguar output file or a GAMESS output file
 * extracts the geometries and energies and prints out the 
 * energies and the distance between successive steps of the
 * minimisation in mass weighted coordinates (Bohr (amu)0.5)
 *
 * If a second argument is given, it will be assumed that this is
 * a .fdt file created by FreqQRC. The geometry and energy in this
 * file will be used as the starting point, instead of the first 
 * structure in the minimisation procedure.
 *
 * These figures can be pasted into Excel to form a quick 
 * alternative to a full IRC calculation. It will not follow
 * the reaction path so precisely but it will identify starting
 * materials and products more rapidly.
 * 
 * The output is recorded in a file called INPUTNAME.qrc
 *
 * @author J M Goodman
 */ 
 
  public static void main(String[] args) { 
/**
 * The names of the input file and the output file
 */
    String versionFindQRC = "1.01";
    String mname, outName; 
    boolean energyUp = false;
    for (int i = 0; i < args.length; i++) {
      if (args[i].equalsIgnoreCase("-u")) {
        energyUp = true;
        if (i == 0 && args.length > 0) {
          for (int ii = 1; ii  < args.length; ii++) {
            args[ii-1] = args[ii];
          }
        } else if (i == 1 && args.length > 2) args[1] = args[2];
      }
    }
    if (args[0].lastIndexOf(".") < 0) { 
      mname = args[0]+".out"; 
      outName = args[0]+".qrc"; 
    } else { 
      mname = args[0]; 
      outName = args[0].substring(0,args[0].lastIndexOf("."))+".qrc"; 
    } 
    boolean fdt = false;
    String fdtFile="fdt";
    if (args.length > 1) {
//    Second argument will be a .fdt file
//    so must read initial energy and geometry from there
      fdt = true;
      fdtFile = args[1];
      System.out.println("Found fdt file: "+fdtFile);
    }
    String fileType = "Jaguar"; 
 
    String writeString; 
    String currLine="XXXX"; 
/**
 * The number of atoms in the structure
 */
    int niatms = 0; 
    BufferedReader buffRead; 
    BufferedWriter buffWrite; 

/**
 * The list of atom types in the molecule
 */ 
    String[] atomtypes; 
/**
 * The list of atomic weights
 */
    double[] atomweight; 
/**
 * The current coordinates of the atoms
 */
    double[] x; 
    double[] y; 
    double[] z; 
    double energy = 0.0; 
    boolean lasttime = false; 

/**
 * ox, oy and oz record the original coordinates of the molecule
 */
    double[] ox; 
    double[] oy; 
    double[] oz; 
    double oenergy = 0.0; 

/**
 * lx, ly and lz record the coordinates of the last structure
 * before each step
 */ 
    double[] lx; 
    double[] ly; 
    double[] lz; 
    double lenergy = 0.0; 
 
    double osos = 0.0; 
    double lsos = 0.0; 
    double totlsos = 0.0; 
    double quatosos = 0.0;
    double quatlsos = 0.0;
    double totquatlsos = 0.0;
 
    System.out.println("FindQRC version "+versionFindQRC);
    System.out.println("(c) J M Goodman, 2003-2004"); 
    System.out.println("Cambridge University"); 
    System.out.println("All rights reserved"); 
    System.out.println(""); 
 
    System.out.println("File name: "+mname); 
 
    try { 
      buffRead = new BufferedReader(new FileReader(new File(mname))); 
 
 
      for (int i = 0; i < 21 ; i++) { 
        currLine = buffRead.readLine(); 
        while (currLine.indexOf("setenv") >= 0) currLine = buffRead.readLine(); 
        if (currLine.indexOf("Jaguar") > -1) fileType = "Jaguar"; 
        if (currLine.indexOf("GAMESS") > -1) fileType = "GAMESS"; 
      } 

//    First read all through the input files, determining whether it is a Jaguar
//    files or a GAMESS file, and counting the atoms
      niatms = 0; 
      if (fileType.equalsIgnoreCase("Jaguar")) { 
        while (currLine.indexOf("Input geometry") < 0) currLine = buffRead.readLine(); 
        currLine = buffRead.readLine(); 
        currLine = buffRead.readLine(); 
        while (currLine.length() > 10) { 
          currLine = buffRead.readLine(); 
          niatms++; 
        } 
        niatms--; 
      } else if (fileType.equalsIgnoreCase("GAMESS")) { 
        while (currLine.indexOf("COORDINATES (BOHR)") < 0) currLine = buffRead.readLine(); 
        currLine = buffRead.readLine(); 
        while (currLine.length() > 10) { 
          currLine = buffRead.readLine(); 
          niatms++; 
        } 
        niatms--; 
        while (currLine.indexOf("LOCATED *****") < 0) currLine = buffRead.readLine(); 
      } 
      buffRead.close(); 
    } catch(Exception e) {;} 

//  Now we know how many atoms there are, set up arrays
    atomtypes = new String[niatms]; 
    atomweight = new double[niatms]; 
    x = new double[niatms]; 
    y = new double[niatms]; 
    z = new double[niatms]; 
 
    ox = new double[niatms]; 
    oy = new double[niatms]; 
    oz = new double[niatms]; 
 
    lx = new double[niatms]; 
    ly = new double[niatms]; 
    lz = new double[niatms]; 
 
 
//  Read through file again, picking up energies and coordinates 
    try { 
     if (fdt) {
//       System.out.println("fdt 1: ");
       buffRead = new BufferedReader(new FileReader(new File(fdtFile)));
       currLine = buffRead.readLine();
//       System.out.println(currLine);
//       System.out.println("01234567890123456789012345678901234567890");
       oenergy = Double.valueOf(currLine.substring(29).trim()).doubleValue();
       for (int i = 0; i < niatms; i++) {
         currLine = buffRead.readLine();
//         System.out.println(currLine);
         if (fileType.equalsIgnoreCase("Jaguar")) {
           atomtypes[i] = currLine.substring(0,9);
           atomweight[i] = atomtype2atomweight(atomtypes[i]);
           ox[i] = Double.valueOf(currLine.substring(9,19).trim()).doubleValue();
           oy[i] = Double.valueOf(currLine.substring(20,32).trim()).doubleValue();
           oz[i] = Double.valueOf(currLine.substring(33).trim()).doubleValue();
         } else {
           atomtypes[i] = currLine.substring(0,9);
           atomweight[i] = atomtype2atomweight(atomtypes[i]);
           ox[i] = Double.valueOf(currLine.substring(17,28).trim()).doubleValue();
           oy[i] = Double.valueOf(currLine.substring(29,40).trim()).doubleValue();
           oz[i] = Double.valueOf(currLine.substring(41).trim()).doubleValue();
         }
       }
       buffRead.close();
     }

      buffRead = new BufferedReader(new FileReader(new File(mname))); 
      buffWrite = new BufferedWriter(new FileWriter(new File(outName))); 
      buffWrite.write("Pseudo-IRC analysis"); 
      buffWrite.newLine(); 
      buffWrite.newLine(); 
       
      currLine = "XXXX"; 
      if (fileType.equalsIgnoreCase("Jaguar")) { 
        while (currLine.indexOf("Input geometry") < 0) currLine = buffRead.readLine(); 
        currLine = buffRead.readLine(); 
        currLine = buffRead.readLine(); 
        for (int i = 0; i < niatms; i++) { 
          currLine = buffRead.readLine(); 
          atomtypes[i] = currLine.substring(0,9); 
          atomweight[i] = atomtype2atomweight(atomtypes[i]); 
          lx[i] = Double.valueOf(currLine.substring(9,25).trim()).doubleValue(); 
          ly[i] = Double.valueOf(currLine.substring(25,43).trim()).doubleValue(); 
          lz[i] = Double.valueOf(currLine.substring(43).trim()).doubleValue(); 
          if (!fdt) {
            ox[i] = lx[i]; 
            oy[i] = ly[i]; 
            oz[i] = lz[i]; 
          }
        } 
      } else if (fileType.equalsIgnoreCase("GAMESS")) { 
        while (currLine.indexOf("COORDINATES (BOHR)") < 0) currLine = buffRead.readLine(); 
        currLine = buffRead.readLine(); 
//      System.out.println(currLine); 
        for (int i = 0; i < niatms; i++) { 
          currLine = buffRead.readLine(); 
          atomtypes[i] = currLine.substring(0,16); 
          atomweight[i] = (double)atomWeightInt[(int)Double.valueOf(atomtypes[i].substring(6).trim()).doubleValue()]; 
          lx[i] = 0.52918*Double.valueOf(currLine.substring(17,33).trim()).doubleValue(); 
          ly[i] = 0.52918*Double.valueOf(currLine.substring(34,53).trim()).doubleValue(); 
          lz[i] = 0.52918*Double.valueOf(currLine.substring(54).trim()).doubleValue(); 
          if (!fdt) {
            ox[i] = lx[i];
            oy[i] = ly[i]; 
            oz[i] = lz[i];
          }
        } 
      } 
 
      if (fileType.equalsIgnoreCase("Jaguar")) { 
//      System.out.println("Looking for energy"); 
        while (currLine.indexOf("Total energy") < 0 && currLine.indexOf("Solution phase energy") < 0) currLine = buffRead.readLine(); 
//      System.out.println("found:"+currLine.substring(38,56)+"::"); 
        lenergy = Double.valueOf(currLine.substring(38,56).trim()).doubleValue(); 
				if (currLine.indexOf("Solution phase energy") >= 0) {
					while (currLine.indexOf("new geometry:") < 0 && currLine.indexOf("final geometry:") < 0) {
						currLine = buffRead.readLine(); 
						if (currLine.indexOf("Solution phase energy") >= 0) {
							lenergy = Double.valueOf(currLine.substring(38,56).trim()).doubleValue();
						}
					}
				}
        if (!fdt) oenergy = lenergy; 
//      System.out.println("Found energy"+oenergy); 
      } else if (fileType.equalsIgnoreCase("GAMESS")) { 
        while (currLine.indexOf("AFTER") < 0 || currLine.indexOf("ENERGY") < 0 ) currLine = buffRead.readLine(); 
        lenergy = Double.valueOf(currLine.substring(currLine.indexOf("ENERGY")+11,currLine.indexOf("AFTER")).trim()).doubleValue(); 
        if (!fdt) oenergy = lenergy;
//        System.out.println("Found energy: "+oenergy); 
      } 
      writeString = "Energy in hartrees; reaction coordinate in bohr (square root of amu)";
      System.out.println(writeString);
      buffWrite.write(writeString);
      buffWrite.newLine();
      writeString = "Energy               Mass-weighted                 Non-mass-weighted"; 
      System.out.println(writeString);
      buffWrite.write(writeString);
      buffWrite.newLine();
      writeString = "                     origin          last structure  origin         last structure";
      System.out.println(writeString);
      buffWrite.write(writeString);
      buffWrite.newLine();

      writeString = doubFormat(oenergy,12,6)+"    "+doubFormat(0.0,12,6)+"    "+doubFormat(0.0,12,6);
      System.out.println(writeString); 
      buffWrite.write(writeString); 
      buffWrite.newLine(); 

      setCentroid(ox, oy, oz, atomweight);
      setCentroid(lx, ly, lz, atomweight);

      if (fdt) {
        quatosos = qtrfitPrep(lx, ly, lz, ox, oy, oz, niatms, atomweight);
//        System.out.println("fdt 3: "+quatosos);
        totquatlsos = quatosos;
        energy = lenergy;
        writeString = doubFormat(energy,12,6)+"    "+doubFormat(quatosos/0.52918,12,6)+"    "+doubFormat(totquatlsos/0.52918,12,6);
        System.out.println(writeString);
        buffWrite.write(writeString);
        buffWrite.newLine();
        buffWrite.flush();
      }

      while (!lasttime) { 
        if (fileType.equalsIgnoreCase("Jaguar")) { 
          while (currLine.indexOf("new geometry:") < 0 && currLine.indexOf("final geometry:") < 0) currLine = buffRead.readLine(); 
          if (currLine.indexOf("final geometry:") >= 0) lasttime = true; 
//          System.out.println("Found geometry  "+lasttime+"  niatms = "+niatms); 
          currLine = buffRead.readLine(); 
          currLine = buffRead.readLine(); 
          for (int i = 0; i < niatms; i++) { 
            currLine = buffRead.readLine(); 
            x[i] = Double.valueOf(currLine.substring(9,25).trim()).doubleValue(); 
            y[i] = Double.valueOf(currLine.substring(25,43).trim()).doubleValue(); 
            z[i] = Double.valueOf(currLine.substring(43).trim()).doubleValue(); 
          } 
          while (currLine.indexOf("Total energy") < 0 && currLine.indexOf("Solution phase energy") < 0) currLine = buffRead.readLine(); 
          energy = Double.valueOf(currLine.substring(38,56).trim()).doubleValue(); 
        } else if (fileType.equalsIgnoreCase("GAMESS")) { 
//          System.out.println("Found geometry  "+lasttime); 
          while (currLine.indexOf("COORDINATES") < 0 && currLine.indexOf("TERMINATED") < 0 || currLine.indexOf("NUCLEAR") >= 0) currLine = buffRead.readLine(); 
          if (currLine.indexOf("TERMINATED") > -1) { 
            lasttime = true; 
          } else { 
            currLine = buffRead.readLine(); 
            currLine = buffRead.readLine(); 
//            System.out.println("currLine: "+currLine); 
            for (int i = 0; i < niatms; i++) { 
              currLine = buffRead.readLine(); 
//              System.out.println("Loop currLine: "+currLine); 
              x[i] = Double.valueOf(currLine.substring(17,32).trim()).doubleValue(); 
              y[i] = Double.valueOf(currLine.substring(33,47).trim()).doubleValue(); 
              z[i] = Double.valueOf(currLine.substring(48).trim()).doubleValue(); 
            } 
            while (currLine.indexOf("AFTER") < 0 || currLine.indexOf("ENERGY") < 0 ) currLine = buffRead.readLine(); 
            energy = Double.valueOf(currLine.substring(currLine.indexOf("ENERGY")+11,currLine.indexOf("AFTER")).trim()).doubleValue(); 
//            System.out.println("Found energy new  "+energy); 
          } 
        } 
 
 
//      Now work out RMS distance for o and l 
        if (energy < lenergy || energyUp) { 
          lenergy = energy; 
          osos = 0.0; 
          lsos = 0.0; 

//        should do rotation here
          setCentroid(x, y, z, atomweight);
          double ttt;
          for (int i = 0; i < niatms; i++) {x[i]=-x[i];y[i]=-y[i];}
          ttt = qtrfitPrep(x, y, z, ox, oy, oz, niatms, atomweight);
          for (int ii = 0; ii < 16; ii++) {
            for (int i = 0; i < niatms; i++) {z[i]=-z[i];y[i]=-y[i];}
            ttt=qtrfitPrep(x, y, z, ox, oy, oz, niatms, atomweight);
            for (int i = 0; i < niatms; i++) {x[i]=-x[i];z[i]=-z[i];}
            ttt=qtrfitPrep(x, y, z, ox, oy, oz, niatms, atomweight);
            for (int i = 0; i < niatms; i++) {x[i]=-x[i];y[i]=-y[i];}
            ttt=qtrfitPrep(x, y, z, ox, oy, oz, niatms, atomweight);
          }

          quatosos = qtrfitPrep(x, y, z, ox, oy, oz, niatms, atomweight);
          for (int i = 0; i < niatms; i++) {
            osos += (x[i]-ox[i])*(x[i]-ox[i]);
            osos += (y[i]-oy[i])*(y[i]-oy[i]);
            osos += (z[i]-oz[i])*(z[i]-oz[i]);
          }
          quatlsos = qtrfitPrep(x, y, z, lx, ly, lz, niatms, atomweight);
          totquatlsos += quatlsos;
          for (int i = 0; i < niatms; i++) { 
            lsos += (x[i]-lx[i])*(x[i]-lx[i]); 
            lsos += (y[i]-ly[i])*(y[i]-ly[i]); 
            lsos += (z[i]-lz[i])*(z[i]-lz[i]); 
          } 
          osos = Math.sqrt(osos); 
          totlsos += Math.sqrt(lsos); 
 
          writeString = doubFormat(energy,12,6)+"    "+doubFormat(quatosos/0.52918,12,6)+"    "+doubFormat(totquatlsos/0.52918,12,6)+"    "+doubFormat(osos/0.52918,12,6)+"    "+doubFormat(totlsos/0.52918,12,6);
          System.out.println(writeString); 
          buffWrite.write(writeString); 
          buffWrite.newLine(); 
          buffWrite.flush(); 

          for (int i = 0; i < niatms; i++) { 
            lx[i] = x[i]; 
            ly[i] = y[i]; 
            lz[i] = z[i]; 
          } 
        } 
      } 
 
      buffRead.close(); 
      buffWrite.close(); 
 
    } catch(Exception e) {;} 
 
  } 
 

/**
 * Create formatted strings for output
 */
  static String doubFormat(double number, int length, int decPlace) { 
    String temp = "          "+Double.toString(number)+"0000000000"; 
    int j = temp.indexOf('.'); 
    temp = temp.substring(0,j+decPlace+1); 
    int i = temp.length(); 
    return temp.substring(i-length); 
  } 

/**
 * Convert atomtype to atomic weight
 */
  static double atomtype2atomweight(String atomname) { 
    int atomnum; 
    String parsename, process; 
    process = atomname.trim(); 
    if (process.length() > 1) { 
      if (Character.isLetter(process.charAt(1))) parsename = process.substring(0,2); 
      else parsename = process.substring(0,1); 
    } else parsename = process; 
//    System.out.println("Atomname: "+atomname+":   :"+process+":   :"+parsename); 
    atomnum = sym2num(parsename); 
    if (atomnum > atomWeightInt.length) System.out.println("Atomic weight too high"); 
    return atomWeightInt[atomnum]; 
  } 

/**
 * Convert atomic symbol to atomic number
 */
  static int sym2num(String symbol) { 
    int atomnum = -1; 
    for (int i=0;i<label.length;i++) { 
      if (symbol.equalsIgnoreCase(label[i])) atomnum = i; 
    } 
    return atomnum; 
  } 
 
/** 
 * Atomic Symbols 
 */ 
  static String label[] = new String[] { " ", 
    "H" , "He", "Li", "Be", "B" , "C" , "N" , "O" , "F" , "Ne", 
    "Na", "Mg", "Al", "Si", "P" , "S" , "Cl", "Ar", 
    "K" , "Ca", "Sc", "Ti", "V" , "Cr", "Mn", "Fe", "Co", 
      "Ni", "Cu", "Zn", "Ga", "Ge", "As", "Se", "Br", "Kr", 
    "Rb", "Sr", "Y", "Zr", "Nb", "Mo", "Tc", "Ru", "Rh", 
      "Pd", "Ag", "Cd", "In", "Sn", "Sb", "Te", "I" , "Xe", 
    "Cs", "Ba", "La", "Ce", "Pr", "Nd", "Pm", "Sm", "Eu", 
      "Gd", "Tb", "Dy", "Ho", "Er", "Tm", "Yb", "Lu", "Hf", 
      "Ta", "W" , "Re", "Os", "Ir", "Pt", "Au", "Hg", 
      "Tl", "Pb", "Bi", "Po", "At", "Rn", 
    "Fr", "Ra", "Ac", "Th", "Pa", "U" , "Np", "Pu", "Am", 
      "Cm", "Bk", "Cf", "Es", "Fm", "Md", "No", "Lr", "Rf", 
      "Db", "Sg", "Bh", "Hs", "Mt" }; 

/**
 * Atomic weights - integer value of main isotope only
 */ 
  static int atomWeightInt[] = new int[] { 0, 
    1, 4, 7, 9, 11, 12, 14, 16, 19, 20, 
    23, 24, 27, 28, 31, 32, 35, 40, 
    39, 40 
  };

/**
 * Before two molecules can be rotated for superimposition, the centroids must 
 * be aligned. This method aligns two conformations of the same molecule. It 
 * assumes all the arrays have length niatms
 */
  public static void setCentroid(double[] x, double[] y, double[] z, double[] atomweight) {
    int niatms = x.length;
    double xsum=0.0, ysum=0.0, zsum=0.0, wsum=0.0;

    for (int i=0; i<niatms; i++) {
      xsum += x[i]*Math.sqrt(atomweight[i]);
      ysum += y[i]*Math.sqrt(atomweight[i]);
      zsum += z[i]*Math.sqrt(atomweight[i]);
      wsum += Math.sqrt(atomweight[i]);
    }
    xsum /= wsum;
    ysum /= wsum;
    zsum /= wsum;
    for (int i=0; i<niatms; i++) {
      x[i] -= xsum;
      y[i] -= ysum;
      z[i] -= zsum;
    }

  }

/**
 * This method prepares the data for the quaternion routine qtrfit and returns the 
 * superposition rms error zz
 */
  public static double qtrfitPrep(double[] x, double[] y, double[] z, double[] ox , double[] oy, double[] oz, int niatms, double[] atomweight) {
    double[][] mb = new double[][] {x, y, z};
    double[][] ma = new double[][] {ox, oy, oz};

    double zz = 0;
    double[] q = new double[4];
    double[][] u = new double[4][4];

    qtrfit(niatms, mb, ma, atomweight, zz, q, u);
    rotate(niatms, u, mb);
    zz = analyse(niatms, atomweight, ma, mb);
//    System.out.println("Final error: "+zz+"  "+Math.sqrt(zz)); 
    return Math.sqrt(zz);
  }
 
/**
 * Rotate a molecule, held in a using matrix held in u
 * Molecule has n atoms
 */
  public static void rotate(int n, double[][] u, double[][] a) {
    double[] t = new double[3];

    for (int i = 0; i < n; i++) {
      t[0] = u[0][0] * a[0][i] + u[0][1] * a[1][i] + u[0][2] * a[2][i];
      t[1] = u[1][0] * a[0][i] + u[1][1] * a[1][i] + u[1][2] * a[2][i];
      t[2] = u[2][0] * a[0][i] + u[2][1] * a[1][i] + u[2][2] * a[2][i];
      a[0][i] = t[0];
      a[1][i] = t[1];
      a[2][i] = t[2];
    }
  }

/**
 * Determine the RMS error between A and B
 */
  public static double analyse(int n, double[] w, double[][] a, double[][] b) {
    double z;
    double[] e = new double[3];

    e[0] = 0.0;
    e[1] = 0.0;
    e[2] = 0.0;
    for (int i = 0; i < n; i++) {
      e[0] = e[0] + w[i] * (a[0][i] - b[0][i]) * (a[0][i] - b[0][i]);
      e[1] = e[1] + w[i] * (a[1][i] - b[1][i]) * (a[1][i] - b[1][i]);
      e[2] = e[2] + w[i] * (a[2][i] - b[2][i]) * (a[2][i] - b[2][i]);
    }
    z = e[0] + e[1] + e[2];
    return z;
  }


/**
 * This routine is a Java translation of the QTRFIT Fortran routine
 * of David Hesiterberg
 *
 * David J. Heisterberg
 * The Ohio Supercomputer Center, 1224 Kinnear Rd,
 * Columbus, OH  43212-1163
 * (614)292-6036; djh@osc.edu
 *
 * Java translation by J M Goodman 2003
 *
 * QTRFIT
 * Find the quaternion, q,[and left rotation matrix, u] 
 * that minimizes
 *   |qTBq - A| ^ 2   [|uB - A| ^ 2]
 * This is equivalent to maximizing Re (qTBTqA).
 * This is equivalent to finding the largest 
 * eigenvalue and corresponding
 * eigenvector of the matrix
 *  [V2   VUx  VUy  VUz ]
 *  [VUx  Ux2  UxUy UzUx]
 *  [VUy  UxUy Uy2  UyUz]
 *  [VUz  UzUx UyUz Uz2 ]
 * where
 *   V2   = Bx Ax + By Ay + Bz Az
 *   Ux2  = Bx Ax - By Ay - Bz Az
 *   Uy2  = By Ay - Bz Az - Bx Ax
 *   Uz2  = Bz Az - Bx Ax - By Ay
 *   VUx  = Bz Ay - By Az
 *   VUy  = Bx Az - Bz Ax
 *   VUz  = By Ax - Bx Ay
 *   UxUy = Bx Ay + By Ax
 *   UyUz = By Az + Bz Ay
 *   UzUx = Bz Ax + Bx Az
 * The left rotation matrix, u, is obtained from q by
 *   u = qT1q
 * INPUT
 *   n      - number of points
 *   b      - molecule to be rotated
 *   a      - reference molecule
 *   w      - weight vector
 * OUTPUT
 *   z      - eigenvalue
 *   q      - the best-fit quaternion
 *   u      - the best-fit left rotation matrix
 *   nr     - number of jacobi sweeps required
 */

  public static void qtrfit (int n, double[][] b, double[][] a, double[] w, double z, double[] q, double[][] u) {
    double xxyx=0.0, xxyy=0.0, xxyz=0.0;
    double xyyx=0.0, xyyy=0.0, xyyz=0.0;
    double xzyx=0.0, xzyy=0.0, xzyz=0.0;
    double[][] c = new double[4][4];
    double[][] v = new double[4][4];
    double[] d = new double[4];
    int i;

// generate the upper triangle of the quadratic form matrix;
    for (i = 0; i < n; i++) {
      xxyx += b[0][i] * a[0][i] * w[i];
      xxyy += b[0][i] * a[1][i] * w[i];
      xxyz += b[0][i] * a[2][i] * w[i];
      xyyx += b[1][i] * a[0][i] * w[i];
      xyyy += b[1][i] * a[1][i] * w[i];
      xyyz += b[1][i] * a[2][i] * w[i];
      xzyx += b[2][i] * a[0][i] * w[i];
      xzyy += b[2][i] * a[1][i] * w[i];
      xzyz += b[2][i] * a[2][i] * w[i];
    }

    c[0][0] = xxyx + xyyy + xzyz;
    c[0][1] = xzyy - xyyz;
    c[1][1] = xxyx - xyyy - xzyz;
    c[0][2] = xxyz - xzyx;
    c[1][2] = xxyy + xyyx;
    c[2][2] = xyyy - xzyz - xxyx;
    c[0][3] = xyyx - xxyy;
    c[1][3] = xzyx + xxyz;
    c[2][3] = xyyz + xzyy;
    c[3][3] = xzyz - xxyx - xyyy;

// diagonalize c;
    qtrjac (c, 4, 4, d, v);

// extract the desired quaternion;
    z = d[3];
    q[0] = v[0][3];
    q[1] = v[1][3];
    q[2] = v[2][3];
    q[3] = v[3][3];

// generate the rotation matrix;
    q2mat (q, u);
  }

/**
 * QTRJAC;
 * Jacobi diagonalizer with sorted output. 
 * Same calling sequence as;
 * EISPACK routine, but must specify nrot!;
 */
  public static void qtrjac (double[][] a, int n, int np, double[] d, double[][] v) {
    double onorm, dnorm;
    double b, dma, q, t, c, s;
    double atemp, vtemp, dtemp;
    int i, j, k, l;
//    int nrot = 16;
    int nrot = 128;

    for (j = 0; j < n; j++) {
      for (i = 0; i < n; i++) {
        v[i][j] = 0.0;
      }
      v[j][j] = 1.0;
      d[j] = a[j][j];
    }

    for (l = 0; l < nrot; l++) {
      dnorm = 0.0;
      onorm = 0.0;
      for (j = 0; j < n; j++) {
        dnorm += Math.abs(d[j]);
        for (i = 0; i < j-1; i++) {
          onorm += Math.abs(a[i][j]);
        }
      }
      if ((onorm + dnorm) <= dnorm) break;
      for (j = 1; j < n; j++) {
        for (i = 0; i < j-1; i++) {
          b = a[i][j];
          if (Math.abs(b) > 0.0) {
            dma = d[j] - d[i];
            if (Math.abs(dma)+Math.abs(b) < Math.abs(dma)) {
              t = b / dma;
            } else {
              q = 0.5 * dma / b;
              t = fortranSign(1.0 / (Math.abs(q) + Math.sqrt(1.0 + q * q)), q);
            }
            c = 1.0 / Math.sqrt(t * t + 1.0);
            s = t * c;
            a[i][j] = 0.0;
            for (k=0; k < i-1; k++) {
              atemp    = c * a[k][i] - s * a[k][j];
              a[k][j] = s * a[k][i] + c * a[k][j];
              a[k][i] = atemp;
            }
            for (k=i+1; k < j; k++) {
              atemp   = c * a[i][k] - s * a[k][j];
              a[k][j] = s * a[i][k] + c * a[k][j];
              a[i][k] = atemp;
            }
            for (k=j; k < n; k++) {
              atemp   = c * a[i][k] - s * a[j][k];
              a[j][k] = s * a[i][k] + c * a[j][k];
              a[i][k] = atemp;
            }
            for (k=0; k < n; k++) {
              vtemp   = c * v[k][i] - s * v[k][j];
              v[k][j] = s * v[k][i] + c * v[k][j];
              v[k][i] = vtemp;
            }
            dtemp = c * c * d[i] + s * s * d[j] - 2.0 * c * s * b;
            d[j] = s * s * d[i] + c * c * d[j] + 2.0 * c * s * b;
            d[i] = dtemp;
          }
        }
      }
    }
    nrot = l;

    for (j = 0; j < n-1; j++) {
      k = j;
      dtemp = d[k];
      for (i = j; i < n; i++) {
        if (d[i] < dtemp) {
          k = i;
          dtemp = d[k];
        }
      }
      if (k > j) {
        d[k] = d[j];
        d[j] = dtemp;
        for (i = 0; i < n; i++) {
          dtemp    = v[i][k];
          v[i][k] = v[i][j];
          v[i][j] = dtemp;
        }
      }
    }
//    System.out.println("Jacobi passes: "+nrot);

  }

/**
 * Q2MAT: Generate a left rotation matrix 
 * from a normalized quaternion;
 * Input:   q      - normalized quaternion;
 * Output:  u      - the rotation matrix;
 */
  public static void q2mat (double[] q, double[][] u) {
    u[0][0] = q[0]*q[0] + q[1]*q[1] - q[2]*q[2] - q[3]*q[3];
    u[1][0] = 2.0 *(q[1] * q[2] - q[0] * q[3]);
    u[2][0] = 2.0 *(q[1] * q[3] + q[0] * q[2]);
    u[0][1] = 2.0 *(q[2] * q[1] + q[0] * q[3]);
    u[1][1] = q[0]*q[0] - q[1]*q[1] + q[2]*q[2] - q[3]*q[3];
    u[2][1] = 2.0 *(q[2] * q[3] - q[0] * q[1]);
    u[0][2] = 2.0 *(q[3] * q[1] - q[0] * q[2]);
    u[1][2] = 2.0 *(q[3] * q[2] + q[0] * q[1]);
    u[2][2] = q[0]*q[0] - q[1]*q[1] - q[2]*q[2] + q[3]*q[3];
  }

/**
 * Java equivalent of Fortran SIGN function
 */
  public static double fortranSign(double a, double b) {
    if (b >= 0) return a;
    else return -a;
  }

}



