import java.io.*;

class FreqQRC {

/**
 * This class reads in a jaguar frequency calculation file 
 * or a GAMESS frequency calculation file
 * and produces a new cartesian file with coordinates with the 
 * first vibrational frequency added, and separately, subtracted.
 * These files can then be minimised as pseudo IRC calculation.
 * 
 * The first step is to find cartesian coordinates describing the final 
 * geometry. These are usually marked 'final geometry' unless the
 * calculation was not a minimisation, in which case they are
 * 'Input geometry'
 *
 * Once these have been read, go on to find frequencies.
 * This program will just read the first - the negative one
 * if there is a negative one.
 */

  public static void main(String[] args) {
		String freqQRCversion = "1.01";
//  Names of the mol file
    double energy=0.0;
    String mname;
    String outName;
    int freqNumber = 1;
    for (int i = 0; i < args.length; i++) {
      if (args[i].indexOf("-f") == 0) {
        if (args[i].length() > 2) {
          if (Character.isDigit(args[i].charAt(2))) {
            freqNumber = Character.getNumericValue(args[i].charAt(2));
          }
        }
        if (i == 0 && args.length > 0) {
          for (int ii = 1; ii  < args.length; ii++) {
            args[ii-1] = args[ii];
          }
        } else if (i == 1 && args.length > 2) args[1] = args[2];
      }
    }
		System.out.println("FreqQRC version"+freqQRCversion);
    if (args[0].lastIndexOf(".") < 0) {
      mname = args[0]+".out";
      outName = args[0]+".fdt";
    } else {
      mname = args[0];
      outName = args[0].substring(0,args[0].lastIndexOf("."))+".fdt";
    }
    String currLine="XXXX";
    int niatms = 0;
    double fmult = 1.0;
    if (args.length > 1 && args[1].indexOf("-f") < 0) {
      fmult = Double.valueOf(args[1]).doubleValue();
//      System.out.println("Fmult = "+fmult);
    }
    BufferedReader buffRead;
    BufferedWriter buffWrite;
    boolean minimised = true;

    String fileType = "Jaguar";

    String[] atomtypes;
    double[] x;
    double[] y;
    double[] z;
    double[] fx;
    double[] fy;
    double[] fz;

    System.out.println("(c) J M Goodman, 2003-2004");
    System.out.println("Cambridge University");
    System.out.println("All rights reserved");
    System.out.println("");

    System.out.println("File name: "+mname);
    System.out.println("Reading frequency number: "+freqNumber);
    System.out.println("Frequency multiplier: "+fmult);

    try {
      buffRead = new BufferedReader(new FileReader(new File(mname)));

      for (int i = 0; i < 10 ; i++) {
        currLine = buffRead.readLine();
        if (currLine.indexOf("Jaguar") > -1) fileType = "Jaguar";
        if (currLine.indexOf("GAMESS") > -1) fileType = "GAMESS";
      }

//    Now set minimised and niatms
      niatms = 0;
      minimised = false;
      if (fileType.equalsIgnoreCase("Jaguar")) {
        while (currLine.indexOf("Input geometry") < 0) currLine = buffRead.readLine();
        currLine = buffRead.readLine();
        currLine = buffRead.readLine();
        while (currLine.length() > 10) {
          currLine = buffRead.readLine();
          niatms++;
        }
        niatms--;
        while (currLine.indexOf("final geometry") < 0) currLine = buffRead.readLine();
        if (currLine.indexOf("final geometry") >= 0) minimised = true;
      } else if (fileType.equalsIgnoreCase("GAMESS")) {
        while (currLine.indexOf("COORDINATES (BOHR)") < 0) currLine = buffRead.readLine();
//        System.out.println("Found: "+currLine);

        currLine = buffRead.readLine();
//        System.out.println("Found: "+currLine);

        while (currLine.length() > 10) {
          currLine = buffRead.readLine();
          niatms++;
        }
        niatms--;
//        System.out.println("Number of atoms: "+niatms);

        while (currLine.indexOf("LOCATED *****") < 0) currLine = buffRead.readLine();
        if (currLine.indexOf("LOCATED *****") >= 0) minimised = true;
//        System.out.println("Number of atoms: "+minimised);

      }
      buffRead.close();
    } catch(Exception e) {;}

    atomtypes = new String[niatms];
    x = new double[niatms];
    y = new double[niatms];
    z = new double[niatms];
    fx = new double[niatms];
    fy = new double[niatms];
    fz = new double[niatms];

   System.out.println("Number of atoms: "+niatms+";  Minimised? "+minimised);

    try {
      buffRead = new BufferedReader(new FileReader(new File(mname)));
      
      currLine = "XXXX";
      if (fileType.equalsIgnoreCase("Jaguar")) {
        if (minimised) while (currLine.indexOf("final geometry") < 0) currLine = buffRead.readLine();
        else while (currLine.indexOf("Input geometry") < 0) currLine = buffRead.readLine();
        currLine = buffRead.readLine();
        currLine = buffRead.readLine();
        for (int i = 0; i < niatms; i++) {
          currLine = buffRead.readLine();
          atomtypes[i] = currLine.substring(0,9);
          x[i] = Double.valueOf(currLine.substring(9,25).trim()).doubleValue();
          y[i] = Double.valueOf(currLine.substring(25,43).trim()).doubleValue();
          z[i] = Double.valueOf(currLine.substring(43).trim()).doubleValue();
//	  System.out.println(currLine);
        }
        while (currLine.indexOf("SCF energy:") < 0) currLine = buffRead.readLine();
        energy = Double.valueOf(currLine.substring(23,42).trim()).doubleValue();
        while (currLine.indexOf("harmonic frequencies") < 0) currLine = buffRead.readLine();
        currLine = buffRead.readLine();
        while (currLine.indexOf("frequencies") < 0) currLine = buffRead.readLine();
        currLine = buffRead.readLine();
//        System.out.println("Number of atoms: "+niatms);
//        System.out.println(currLine);
        for (int i = 0; i < niatms; i++) {
          currLine = buffRead.readLine();
          if (currLine.indexOf("reduc. mass") > -1) currLine = buffRead.readLine();
//	System.out.println(currLine.substring(4+9*freqNumber,14+9*freqNumber).trim());
          fx[i] = Double.valueOf(currLine.substring(4+9*freqNumber,14+9*freqNumber).trim()).doubleValue();
//        System.out.println("x: "+niatms+"  "+fx[i]);
          currLine = buffRead.readLine();
          fy[i] = Double.valueOf(currLine.substring(4+9*freqNumber,14+9*freqNumber).trim()).doubleValue();
//        System.out.println("y: "+niatms+"  "+fy[i]);
          currLine = buffRead.readLine();
          fz[i] = Double.valueOf(currLine.substring(4+9*freqNumber,14+9*freqNumber).trim()).doubleValue();
//        System.out.println("z: "+niatms+"  "+fz[i]);
        }
      } else if (fileType.equalsIgnoreCase("GAMESS")) {
//        System.out.println("GAMESS file");
        if (minimised) while (currLine.indexOf("LOCATED *****") < 0) currLine = buffRead.readLine();
        else while (currLine.indexOf("COORDINATES (BOHR)") < 0) currLine = buffRead.readLine();
//        System.out.println("GAMESS file  "+minimised);

        currLine = buffRead.readLine();
        if (minimised) {
          currLine = buffRead.readLine();
          currLine = buffRead.readLine();
          for (int i = 0; i < niatms; i++) {
            currLine = buffRead.readLine();
            atomtypes[i] = currLine.substring(0,9);
            x[i] = Double.valueOf(currLine.substring(17,31).trim()).doubleValue();
            y[i] = Double.valueOf(currLine.substring(32,46).trim()).doubleValue();
            z[i] = Double.valueOf(currLine.substring(47).trim()).doubleValue();
          }
        } else {
//        System.out.println("GAMESS file not minimised");
//        System.out.println(currLine);
          for (int i = 0; i < niatms; i++) {
            currLine = buffRead.readLine();
            atomtypes[i] = currLine.substring(0,16);
            x[i] = 0.52918*Double.valueOf(currLine.substring(17,33).trim()).doubleValue();
            y[i] = 0.52918*Double.valueOf(currLine.substring(34,53).trim()).doubleValue();
            z[i] = 0.52918*Double.valueOf(currLine.substring(54).trim()).doubleValue();
          }
        }
        while (currLine.indexOf("FREQUENCY") < 0) currLine = buffRead.readLine();
        currLine = buffRead.readLine();
//        currLine = buffRead.readLine();
				
//        System.out.println("Number of atoms: "+niatms);
//        System.out.println(currLine);
        for (int i = 0; i < niatms; i++) {
//          currLine = buffRead.readLine();
          while (currLine.indexOf("X") < 0) currLine = buffRead.readLine();
          fx[i] = Double.valueOf(currLine.substring(9+12*freqNumber,20+12*freqNumber).trim()).doubleValue();
//        System.out.println("x: "+niatms+"  "+fx[i]);
          currLine = buffRead.readLine();
          fy[i] = Double.valueOf(currLine.substring(9+12*freqNumber,20+12*freqNumber).trim()).doubleValue();
//        System.out.println("y: "+niatms+"  "+fy[i]);
          currLine = buffRead.readLine();
          fz[i] = Double.valueOf(currLine.substring(9+12*freqNumber,20+12*freqNumber).trim()).doubleValue();
//        System.out.println("z: "+niatms+"  "+fz[i]);
        }


      }
      buffRead.close();
    } catch(Exception e) {;}

    try {
      buffWrite = new BufferedWriter(new FileWriter(new File(outName)));
			buffWrite.write("File name: "+mname);
      buffWrite.newLine();
      buffWrite.write("Reading frequency number: "+freqNumber);
      buffWrite.newLine();
      buffWrite.write("Frequency multiplier: "+fmult);
      buffWrite.newLine();

      buffWrite.write("Initial geometry and Energy:    "+energy);
      buffWrite.newLine();
      for (int i = 0; i < niatms; i++) {
        buffWrite.write(atomtypes[i]+doubFormat(x[i],10,6)+"  "+doubFormat(y[i],10,6)+"  "+doubFormat(z[i],10,6));
        buffWrite.newLine();
      }
      buffWrite.newLine();
      buffWrite.newLine();
      buffWrite.write("changes");
      buffWrite.newLine();
      for (int i = 0; i < niatms; i++) {
        buffWrite.write(atomtypes[i]+doubFormat(fx[i],10,6)+"  "+doubFormat(fy[i],10,6)+"  "+doubFormat(fz[i],10,6));
        buffWrite.newLine();
      }
      buffWrite.newLine();
      buffWrite.newLine();
      buffWrite.write("Adding");
      buffWrite.newLine();
      for (int i = 0; i < niatms; i++) {
        buffWrite.write(atomtypes[i]+" "+doubFormat(fmult*fx[i]+x[i],10,6)+"  "+doubFormat(fmult*fy[i]+y[i],10,6)+"  "+doubFormat(fmult*fz[i]+z[i],10,6));
        buffWrite.newLine();
//	System.out.println(fx[i]+"    "+(fmult*fx[i]+x[i])+"    "+doubFormat(fmult*fx[i]+x[i],10,6));
      }
      buffWrite.newLine();
      buffWrite.newLine();
      buffWrite.write("Subtracting");
      buffWrite.newLine();
      for (int i = 0; i < niatms; i++) {
        buffWrite.write(atomtypes[i]+" "+doubFormat(-fmult*fx[i]+x[i],10,6)+"  "+doubFormat(-fmult*fy[i]+y[i],10,6)+"  "+doubFormat(-fmult*fz[i]+z[i],10,6));
        buffWrite.newLine();
      }

      buffWrite.close();
    } catch(Exception e) {;}
  }


  static String doubFormat(double number, int length, int decPlace) {
//	System.out.println(number);
    String temp = "          0.000000000000000";
    String temp2 = Double.toString(number);
    int k = temp2.indexOf("E");
    if (k < 0) {
      temp = "          "+Double.toString(number)+"0000000000";
      int j = temp.indexOf('.');
      temp = temp.substring(0,j+decPlace+1);
      int i = temp.length();
      return temp.substring(i-length);
    } else {
      temp2 = temp2.substring(k);
      int m = temp2.length();
      if (m > decPlace) {
        temp = "          0.000000000000000";
        int j = temp.indexOf('.');
        temp = temp.substring(0,j+decPlace+1);
      } else {
        temp = "          "+Double.toString(number).substring(0,k)+"0000000000000000";
        int j = temp.indexOf('.');
        temp = temp.substring(0,j+decPlace+1-m)+temp2;
      }
      int i = temp.length();
      return temp.substring(i-length);
    }
  }


}


